#pragma once

#include <stdint.h>
#include "Errors.h"
#include "Defines.h"

#ifdef __cplusplus
extern "C" {
#endif

struct SecureStoreNFC_Session;

/**
 * @brief Initializes SecureStore
 * 
 * @param ssDir A directory which is accessible for writing. It will be used for logs.
 */
void SecureStoreNFC_Initialize(const char *ssDir);

/**
 * @brief Creates a session for communicating with a card.
 * 
 * @param commFn A function used for communicating with the card. Parameters are:
 * A void* supplied as @p userPtr
 * Pointer to the data to be sent to the card
 * The length in bytes of the data to be sent to the card
 * Pointer to a buffer that will receive the card's response
 * Pointer to a number that will contain the size of the receiving buffer on call, and will be assigned the actual length of the data received from the card
 * The function returns a 32bit signed number that might be returned from SCardTransmit. On success should be SCARD_S_SUCCESS (0),
 * failure may be indicated by many values, but SCARD_E_UNEXPECTED(0x8010001F) can be used for all of them.
 *
 * @param userPtr The pointer that will be passed as the first parameter to @p commFn
 * @param[out] session A pointer that will receive the session pointer on successful completion of the function
 * @return SecureStoreErr The result status
 */
SecureStoreErr SecureStoreNFC_NewSession(int32_t(*commFn)(void*, const uint8_t*, uint32_t, uint8_t*, uint32_t*), void *userPtr, SecureStoreNFC_Session **session);

/**
 * @brief Releases a session previously created by SecureStoreNFC_NewSession
 * 
 * @param session The session to be released
 */
void SecureStoreNFC_FreeSession(SecureStoreNFC_Session *session);

/**
 * @brief Establishes a PACE-protected channel with the card. Necessary for all card operations except reading the chip ID.
 * If this is not done, functions requiring authentication should return ERR_PACE_NOT_AUTHENTICATED
 * @param session The session to be authenticated
 * @param can The Card Access Number. Should be six decimal digits printed on the card.
 * @return SecureStoreErr The authentication status. If the CAN is wrong, should return ERR_CARD_AUTH_CODE_INVALID
 */
SecureStoreErr SecureStoreNFC_AuthenticatePace(SecureStoreNFC_Session *session, const char *can);

/**
 * @brief Reads the chip ID, which can be used to identify a card for the purpose of CAN caching.
 * Doesn't require PACE authentication.
 * @param session A previously obtained session
 * @param buf A buffer that will receive the chip ID. Should be 8 bytes.
 * @param len A pointer to the length of the buffer. On return (even with error ERR_BUFFER_TOO_SMALL), will receive the actual length of the chip ID
 * @return SecureStoreErr The result status
 */
SecureStoreErr SecureStoreNFC_ReadChipId(SecureStoreNFC_Session *session, uint8_t *buf, uint32_t *len);

/**
 * @brief Loads the card contents and returns them as JSON
 * 
 * @param session A previously obtained session
 * @param cardInfo A pointer that will receive a pointer to the C-string containing the resulting JSON. Releasing it using free() is the caller's responsibility
 * @return SecureStoreErr The result status
 */
SecureStoreErr SecureStoreNFC_LoadCard(SecureStoreNFC_Session *session, char **cardInfo);

/**
 * @brief Logins to the card. This is necessary for things such as private key operations.
 * Card operations not requiring @p authId to be logged in may invalidate the login.
 * 
 * @param session A session to be logged in
 * @param authId The auth id corresponding to the private key you want to work with
 * @param authCode The auth code
 * @param authCodeLen The length of the auth code in bytes
 * @param[out] remainingAttempts A pointer to an uint32, which will receive the number of remaining unsuccessful attempts
 * before @p authId will be blocked. May be null if the information is not desired.
 * @return SecureStoreErr The result status
 */
SecureStoreErr SecureStoreNFC_Login(
	SecureStoreNFC_Session *session,
	SecureStoreCardAuthId authId,
	uint8_t *authCode,
	uint32_t authCodeLen,
	uint32_t *remainingAttempts
);

/**
 * @brief Signs a hash
 * 
 * @param session A previously obtained session that is logged in with the key's authorizationPin
 * @param keyId The id of the key to be used for the signature
 * @param hashType The algorithm used to generate @p hash
 * @param hash The hash
 * @param hashLen The length of @p hash in bytes
 * @param signatureBuf If it points to a null pointer, will receive a buffer allocated by the function, which must be released using free(). Otheriwse *signatureBuf
 * must point to a buffer that will receive the signature.
 * @param[in,out] signatureLen Upon successful completion, is set to the length of the signature in bytes. If @p *signatureBuf != null,
 * caller must initialize the pointee to the size of the supplied buffer.
 * @return SecureStoreErr The result status
 */
SecureStoreErr SecureStoreNFC_SignHash(
	SecureStoreNFC_Session *session,
	uint32_t keyId,
	SecureStoreHashType hashType,
	SecureStoreSignatureScheme sigScheme,
	uint8_t *hash,
	uint32_t hashLen,
	uint8_t **signatureBuf,
	uint32_t *signatureLen
);

/**
 * @brief Uninitializes SecureStore
 * 
 */
void SecureStoreNFC_Uninitialize();

#ifdef __cplusplus
}
#endif
