package cz.ica.icacard

open class SecureStoreException(msg: String, errNumber: Int, cause: Throwable?): Exception(msg, cause) {
    val errNumber = errNumber
    constructor(msg: String, errNumber: Int): this(msg, errNumber, null)
    constructor(msg: String, cause: Throwable): this(msg, 0, cause)
}

class InvalidPacePinException: SecureStoreException("Incorrect PACE PIN", 124)

class InvalidPinException: SecureStoreException("Incorrect PIN", 124)

enum class CardAuthId {
    Pin,
    ESignPin1,
}

enum class HashType {
    Md4,
    Md5,
    Sha1,
    Sha224,
    Sha256,
    Sha384,
    Sha512,
    Ssl3,
}

enum class SignatureScheme {
    RsaPkcs1,
    RsaPss,
    Ecc,
}

class ICACommSDK(sendFn: (ByteArray) -> ByteArray) {
    public external fun Initialize(ssDir: String);
    public external fun NewSession(): Long;
    public external fun FreeSession(session: Long);
    public external fun AuthenticatePace(session: Long, pin: String);
    public external fun ReadChipId(session: Long): ByteArray;
    public external fun LoadCard(session: Long): String;
    public external fun Login(session: Long, authId: CardAuthId, authCode: ByteArray, remainingAttempts: IntArray)
    public external fun SignHash(session: Long, keyId: Int, hashType: HashType, sigScheme: SignatureScheme, hash: ByteArray): ByteArray
    public external fun Uninitialize();
    public val doSend = sendFn
    private fun SendAPDU(data: ByteArray): ByteArray {
        return doSend(data)
    }
    companion object {
        init {
            System.loadLibrary("ICASecureStoreNFCComm")
        }
        public fun CardAuthIdFromString(str: String): CardAuthId {
            return when(str) {
                "Pin" -> CardAuthId.Pin
                "EsignPin1" -> CardAuthId.ESignPin1
                else -> throw RuntimeException("Unexpected CardAuthId value ${str}")
            }
        }
    }
    
}